<?php
session_start();
require_once "../includes/role_helpers.php";
requireLogin();
require_once "../config/config.php";

$conn = getDbConnection();

$success_msg = $error_msg = "";

// Insert new prescription (sales_rep or admin)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $patient_name = trim($_POST['patient_name'] ?? '');
    $product_id = intval($_POST['product_id'] ?? 0);
    $quantity = intval($_POST['quantity'] ?? 0);
    $notes = trim($_POST['notes'] ?? '');

    if ($patient_name && $product_id > 0 && $quantity > 0) {
        $stmt = $conn->prepare("INSERT INTO prescriptions (user_id, patient_name, product_id, quantity, notes, prescribed_at) VALUES (?, ?, ?, ?, ?, NOW())");
        $user_id = $_SESSION['user_id'];
        $stmt->bind_param("isiss", $user_id, $patient_name, $product_id, $quantity, $notes);
        if ($stmt->execute()) {
            $success_msg = "Prescription saved.";
        } else {
            $error_msg = "Error saving prescription.";
        }
        $stmt->close();
    } else {
        $error_msg = "Fill all required fields.";
    }
}

// Fetch prescriptions by logged-in user if sales_rep, else all for admin
if (isAdmin()) {
    $prescriptions = $conn->query("SELECT pr.*, p.product_name, u.username FROM prescriptions pr JOIN products p ON pr.product_id = p.product_id JOIN users u ON pr.user_id = u.user_id ORDER BY prescribed_at DESC");
} else {
    $stmt = $conn->prepare("SELECT pr.*, p.product_name FROM prescriptions pr JOIN products p ON pr.product_id = p.product_id WHERE pr.user_id = ? ORDER BY prescribed_at DESC");
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $prescriptions = $stmt->get_result();
}

$products = $conn->query("SELECT product_id, product_name FROM products ORDER BY product_name");
$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Prescriptions</title>
    <link rel="stylesheet" href="css/bootstrap.min.css" />
    <style>.container{max-width:900px;margin:30px auto;}</style>
</head>
<body>
<div class="container">
    <h2>Prescriptions</h2>
    <?php if ($success_msg): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success_msg) ?></div>
    <?php elseif ($error_msg): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error_msg) ?></div>
    <?php endif; ?>

    <form method="post" class="mb-4">
        <div class="form-row">
            <div class="col-md-4">
                <input type="text" name="patient_name" placeholder="Patient Name" class="form-control" required />
            </div>
            <div class="col-md-4">
                <select name="product_id" class="form-control" required>
                    <option value="">Select Product</option>
                    <?php while($prod = $products->fetch_assoc()): ?>
                        <option value="<?= $prod['product_id'] ?>"><?= htmlspecialchars($prod['product_name']) ?></option>
                    <?php endwhile; ?>
                </select>
            </div>
            <div class="col-md-2">
                <input type="number" name="quantity" placeholder="Quantity" min="1" class="form-control" required />
            </div>
            <div class="col-md-2">
                <button type="submit" class="btn btn-primary btn-block">Save</button>
            </div>
        </div>
        <div class="form-group mt-2">
            <textarea name="notes" placeholder="Notes (optional)" class="form-control" rows="2"></textarea>
        </div>
    </form>

    <h4>Past Prescriptions</h4>
    <table class="table table-bordered table-striped">
        <thead><tr>
            <th>Patient Name</th>
            <th>Product</th>
            <th>Quantity</th>
            <th>Notes</th>
            <th>Prescribed At</th>
            <?php if (isAdmin()): ?><th>By User</th><?php endif; ?>
        </tr></thead>
        <tbody>
        <?php while ($row = $prescriptions->fetch_assoc()): ?>
            <tr>
                <td><?= htmlspecialchars($row['patient_name']) ?></td>
                <td><?= htmlspecialchars($row['product_name']) ?></td>
                <td><?= htmlspecialchars($row['quantity']) ?></td>
                <td><?= htmlspecialchars($row['notes']) ?></td>
                <td><?= htmlspecialchars($row['prescribed_at']) ?></td>
                <?php if (isAdmin()): ?><td><?= htmlspecialchars($row['username']) ?></td><?php endif; ?>
            </tr>
        <?php endwhile; ?>
        </tbody>
    </table>
</div>
</body>
</html>
