<?php
session_start();
require_once "../includes/role_helpers.php";
requireAdmin();
require_once "../config/config.php";
require_once __DIR__ . '/../vendor/autoload.php';

use Picqer\Barcode\BarcodeGeneratorPNG;

$conn = getDbConnection();

$generator = new BarcodeGeneratorPNG();

// Fetch categories for dropdown
$categories = $conn->query("SELECT category_id, category_name FROM categories ORDER BY category_name");

// Initialize messages
$success_msg = $error_msg = "";

// Handle product add/update/delete
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_id = $_POST['product_id'] ?? null;
    $name = trim($_POST['product_name'] ?? '');
    $price = floatval($_POST['price'] ?? 0);
    $category_id = intval($_POST['category_id'] ?? 0);
    $barcode = trim($_POST['barcode'] ?? '');

    if ($name && $price > 0 && $category_id > 0 && $barcode !== '') {
        if ($product_id) {
            // Update product including barcode
            $stmt = $conn->prepare("UPDATE products SET product_name = ?, price = ?, category_id = ?, barcode = ? WHERE product_id = ?");
            $stmt->bind_param("sdisi", $name, $price, $category_id, $barcode, $product_id);
            $stmt->execute();
            $stmt->close();
            $success_msg = "Product updated.";
        } else {
            // Insert product including barcode
            $stmt = $conn->prepare("INSERT INTO products (product_name, price, category_id, barcode) VALUES (?, ?, ?, ?)");
            $stmt->bind_param("sdis", $name, $price, $category_id, $barcode);
            $stmt->execute();
            $stmt->close();
            $success_msg = "Product added.";
        }
    } else {
        $error_msg = "Fill all fields with valid data including barcode.";
    }
}

if (isset($_GET['delete'])) {
    $del_id = intval($_GET['delete']);
    $stmt = $conn->prepare("DELETE FROM products WHERE product_id = ?");
    $stmt->bind_param("i", $del_id);
    $stmt->execute();
    $stmt->close();
    $success_msg = "Product deleted.";
}

// Fetch products with category and barcode
$sql = "SELECT p.product_id, p.product_name, p.price, p.barcode, c.category_name, p.category_id
        FROM products p
        LEFT JOIN categories c ON p.category_id = c.category_id 
        ORDER BY p.product_name";
$products = $conn->query($sql);

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Products - Admin</title>
    <link rel="stylesheet" href="css/bootstrap.min.css" />
    <style>.container{max-width:900px;margin:30px auto;}</style>
</head>
<body>
<div class="container">
    <h2>Product Management</h2>
    <?php if ($success_msg): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success_msg) ?></div>
    <?php elseif ($error_msg): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error_msg) ?></div>
    <?php endif; ?>

    <form method="post" class="mb-4 row g-2 align-items-center">
        <input type="hidden" name="product_id" id="product_id" />
        <div class="col-md-3">
            <input type="text" name="product_name" id="product_name" placeholder="Product Name" class="form-control" required />
        </div>
        <div class="col-md-2">
            <input type="number" name="price" id="price" placeholder="Price" step="0.01" min="0" class="form-control" required />
        </div>
        <div class="col-md-3">
            <select name="category_id" id="category_id" class="form-control" required>
                <option value="">Select Category</option>
                <?php while ($cat = $categories->fetch_assoc()): ?>
                    <option value="<?= $cat['category_id'] ?>"><?= htmlspecialchars($cat['category_name']) ?></option>
                <?php endwhile; ?>
            </select>
        </div>
        <div class="col-md-3">
            <input type="text" name="barcode" id="barcode" placeholder="Barcode" class="form-control" required />
        </div>
        <div class="col-md-1 d-grid">
            <button type="submit" class="btn btn-primary">Save</button>
        </div>
    </form>

    <table class="table table-bordered table-striped align-middle">
        <thead>
            <tr>
                <th>Name</th>
                <th>Price</th>
                <th>Category</th>
                <th>Barcode</th>
                <th>Barcode Image</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
        <?php while ($prod = $products->fetch_assoc()): ?>
            <tr>
                <td><?= htmlspecialchars($prod['product_name']) ?></td>
                <td><?= number_format($prod['price'], 2) ?></td>
                <td><?= htmlspecialchars($prod['category_name']) ?></td>
                <td><?= htmlspecialchars($prod['barcode']) ?></td>
                <td>
                    <?php if (!empty($prod['barcode'])): ?>
                        <img src="data:image/png;base64,<?= base64_encode($generator->getBarcode($prod['barcode'], $generator::TYPE_CODE_128, 2, 50)) ?>" alt="Barcode" />
                    <?php else: ?>
                        N/A
                    <?php endif; ?>
                </td>
                <td>
                    <button class="btn btn-sm btn-info edit-btn" 
                            data-id="<?= $prod['product_id'] ?>" 
                            data-name="<?= htmlspecialchars($prod['product_name'], ENT_QUOTES) ?>"
                            data-price="<?= $prod['price'] ?>"
                            data-categoryid="<?= $prod['category_id'] ?? '' ?>"
                            data-barcode="<?= htmlspecialchars($prod['barcode'], ENT_QUOTES) ?>"
                            >Edit</button>
                    <a href="?delete=<?= $prod['product_id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('Delete this product?');">Delete</a>
                </td>
            </tr>
        <?php endwhile; ?>
        </tbody>
    </table>
</div>

<script>
document.querySelectorAll('.edit-btn').forEach(button => {
    button.addEventListener('click', () => {
        document.getElementById('product_id').value = button.dataset.id;
        document.getElementById('product_name').value = button.dataset.name;
        document.getElementById('price').value = button.dataset.price;
        document.getElementById('category_id').value = button.dataset.categoryid;
        document.getElementById('barcode').value = button.dataset.barcode;
    });
});
</script>
</body>
</html>
