<?php
session_start();
require_once "../includes/role_helpers.php";
requireAdmin();
require_once "../config/config.php";

$conn = getDbConnection();

$from_date = $_POST['from_date'] ?? '';
$to_date = $_POST['to_date'] ?? '';

$whereClause = "";
$params = [];
$types = "";

// Initialize result variables to avoid undefined warnings
$totalSales = ['total_sales' => 0, 'total_revenue' => 0];
$productSalesResult = null;

if (!empty($from_date) && !empty($to_date)) {
    $from_date_db = date('Y-m-d', strtotime($from_date));
    $to_date_db = date('Y-m-d', strtotime($to_date));
    $whereClause = "WHERE DATE(s.sale_date) BETWEEN ? AND ?";
    $params = [$from_date_db, $to_date_db];
    $types = "ss";
}

// Total sales and revenue (sum quantity * unit_price)
$sql_total = "
    SELECT COUNT(DISTINCT s.sale_id) AS total_sales, 
           IFNULL(SUM(si.quantity * si.unit_price), 0) AS total_revenue
    FROM sales s
    JOIN sale_items si ON s.sale_id = si.sale_id
    $whereClause
";
$stmt_total = $conn->prepare($sql_total);
if (!$stmt_total) {
    die("Prepare failed for total sales query: " . $conn->error);
}
if ($whereClause) {
    $stmt_total->bind_param($types, ...$params);
}
$stmt_total->execute();
$result_total = $stmt_total->get_result();
if ($result_total) {
    $totalSales = $result_total->fetch_assoc();
}
$stmt_total->close();

// Sales by product: sum quantity and revenue per product
$sql_product = "
    SELECT p.product_name, 
           SUM(si.quantity) AS total_quantity, 
           SUM(si.quantity * si.unit_price) AS revenue
    FROM sale_items si
    JOIN products p ON si.product_id = p.product_id
    JOIN sales s ON si.sale_id = s.sale_id
    $whereClause
    GROUP BY si.product_id
    ORDER BY revenue DESC
";
$stmt_product = $conn->prepare($sql_product);
if (!$stmt_product) {
    die("Prepare failed for product sales query: " . $conn->error);
}
if ($whereClause) {
    $stmt_product->bind_param($types, ...$params);
}
$stmt_product->execute();
$productSalesResult = $stmt_product->get_result();
$stmt_product->close();

// Monthly sales and revenue for chart
$sql_monthly = "
    SELECT DATE_FORMAT(s.sale_date, '%Y-%m') AS sales_month,
           COUNT(DISTINCT s.sale_id) AS sales_count,
           SUM(si.quantity * si.unit_price) AS revenue
    FROM sales s
    JOIN sale_items si ON s.sale_id = si.sale_id
    $whereClause
    GROUP BY sales_month
    ORDER BY sales_month
";
$stmt_monthly = $conn->prepare($sql_monthly);
if (!$stmt_monthly) {
    die("Prepare failed for monthly sales query: " . $conn->error);
}
if ($whereClause) {
    $stmt_monthly->bind_param($types, ...$params);
}
$stmt_monthly->execute();
$monthlyResult = $stmt_monthly->get_result();

$chartLabels = [];
$chartSales = [];
$chartRevenue = [];
if ($monthlyResult) {
    while ($row = $monthlyResult->fetch_assoc()) {
        $chartLabels[] = $row['sales_month'];
        $chartSales[] = (int)$row['sales_count'];
        $chartRevenue[] = (float)$row['revenue'];
    }
}
$stmt_monthly->close();

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Sales Reports - Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        body {
            background: #f1f3f5;
            min-height: 100vh;
        }
        .container {
            max-width: 960px;
            margin: 40px auto 80px;
            padding: 0 20px;
        }
        h2 {
            font-weight: 700;
            margin-bottom: 30px;
            color: #222;
            text-align: center;
        }
        form.form-inline {
            display: flex;
            gap: 12px;
            flex-wrap: wrap;
            justify-content: center;
            margin-bottom: 30px;
        }
        form.form-inline label {
            align-self: center;
            font-weight: 600;
            margin-bottom: 0;
        }
        .btn {
            min-width: 100px;
        }
        .summary-card {
            background: white;
            padding: 20px 30px;
            border-radius: 15px;
            box-shadow: 0 6px 18px rgba(0,0,0,0.08);
            margin-bottom: 40px;
            text-align: center;
        }
        .summary-card h5 {
            margin-bottom: 20px;
            font-weight: 700;
            color: #0d6efd;
        }
        .summary-card p {
            font-size: 1.2rem;
            margin-bottom: 6px;
        }
        h5.section-title {
            margin-bottom: 20px;
            color: #0d6efd;
            font-weight: 700;
        }
        table.table {
            background: white;
            border-radius: 10px;
            box-shadow: 0 3px 12px rgba(0,0,0,0.05);
        }
        table.table thead tr {
            background-color: #e7f1ff;
            color: #0156d3;
            font-weight: 600;
            border-bottom: 3px solid #0d6efd;
        }
        table.table tbody tr:hover {
            background-color: #f4faff;
        }
        .chart-container {
            background: white;
            padding: 20px 25px;
            border-radius: 15px;
            box-shadow: 0 6px 18px rgba(0,0,0,0.08);
            margin-bottom: 50px;
        }
        @media print {
            body * {
                visibility: hidden;
            }
            .container, .container * {
                visibility: visible;
            }
            .container {
                position: absolute;
                left: 0;
                top: 0;
                width: 100%;
                background: #fff;
                color: #000;
                font-size: 12pt;
                padding: 20px;
            }
            button, form, .btn, .form-control {
                display: none !important;
            }
            canvas {
                max-width: 100% !important;
                height: auto !important;
            }
        }
    </style>
</head>
<body>
<div class="container">
    <h2>Sales Reports</h2>

    <form method="post" action="reports.php" class="form-inline">
        <label for="from_date">From:</label>
        <input type="date" id="from_date" name="from_date" value="<?= htmlspecialchars($from_date) ?>" class="form-control" required />

        <label for="to_date">To:</label>
        <input type="date" id="to_date" name="to_date" value="<?= htmlspecialchars($to_date) ?>" class="form-control" required />

        <button type="submit" class="btn btn-primary">Filter</button>
        <a href="reports.php" class="btn btn-secondary">Reset</a>
    </form>

    <div class="text-center mb-4">
        <button onclick="window.print()" class="btn btn-outline-secondary">Print Report</button>
    </div>

    <div class="summary-card">
        <h5>Overall Summary</h5>
        <p>Total Sales Transactions: <strong><?= htmlspecialchars($totalSales['total_sales']) ?></strong></p>
        <p>Total Revenue: <strong>GH₵ <?= number_format($totalSales['total_revenue'], 2) ?></strong></p>
    </div>

    <div class="chart-container">
        <h5 class="section-title">Monthly Sales & Revenue</h5>
        <canvas id="salesChart" height="100"></canvas>
    </div>

    <h5 class="section-title">Sales by Product</h5>
    <table class="table table-bordered table-striped">
        <thead>
        <tr>
            <th>Product Name</th>
            <th>Total Quantity Sold</th>
            <th>Revenue (GH₵)</th>
        </tr>
        </thead>
        <tbody>
        <?php
        if ($productSalesResult):
            while ($row = $productSalesResult->fetch_assoc()): ?>
                <tr>
                    <td><?= htmlspecialchars($row['product_name']) ?></td>
                    <td><?= htmlspecialchars($row['total_quantity']) ?></td>
                    <td><?= number_format($row['revenue'], 2) ?></td>
                </tr>
            <?php endwhile;
        else: ?>
            <tr><td colspan="3" class="text-center">No sales data found.</td></tr>
        <?php endif; ?>
        </tbody>
    </table>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    const ctx = document.getElementById('salesChart').getContext('2d');
    const salesChart = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: <?= json_encode($chartLabels) ?>,
            datasets: [
                {
                    label: 'Sales Count',
                    data: <?= json_encode($chartSales) ?>,
                    backgroundColor: 'rgba(54, 162, 235, 0.8)',
                    borderRadius: 4,
                },
                {
                    label: 'Revenue',
                    data: <?= json_encode($chartRevenue) ?>,
                    backgroundColor: 'rgba(255, 206, 86, 0.8)',
                    borderRadius: 4,
                }
            ]
        },
        options: {
            scales: {
                y: {
                    beginAtZero: true,
                    grid: { drawBorder: false }
                },
                x: {
                    grid: { display: false }
                }
            },
            responsive: true,
            interaction: { mode: 'index', intersect: false },
            plugins: {
                legend: {
                    labels: { boxWidth: 20, padding: 15 }
                },
                tooltip: {
                    mode: 'index',
                    intersect: false
                }
            }
        }
    });
</script>
</body>
</html>
